<?php
declare(strict_types=1);

require_once __DIR__ . '/wp_common.php';
require_once __DIR__ . '/content_drafts.php';

function wp_master_key_file(array $cfg): string {
  $path = trim((string)($cfg['wp_master_key_b64'] ?? ''));
  if ($path === '') {
    throw new RuntimeException('Missing wp_master_key_b64 configuration');
  }
  return $path;
}

function wp_master_key(string $path): string {
  if (!is_file($path)) {
    @mkdir(dirname($path), 0750, true);
    @file_put_contents($path, base64_encode(random_bytes(32)), LOCK_EX);
    @chmod($path, 0640);
  }
  $b64 = trim((string)@file_get_contents($path));
  $key = base64_decode($b64, true);
  if (!is_string($key) || strlen($key) !== 32) {
    throw new RuntimeException('Invalid wp master key file');
  }
  return $key;
}

function wp_decrypt_password(array $data, string $key): string {
  $enc = trim((string)($data['app_pass_enc'] ?? ''));
  $iv = trim((string)($data['iv'] ?? ''));
  if ($enc === '' || $iv === '') {
    throw new RuntimeException('Missing encrypted credentials');
  }
  $cipher = base64_decode($enc, true);
  $ivRaw = base64_decode($iv, true);
  if (!is_string($cipher) || !is_string($ivRaw) || strlen($ivRaw) !== 16) {
    throw new RuntimeException('Bad encrypted payload');
  }
  $pass = openssl_decrypt($cipher, 'aes-256-cbc', $key, OPENSSL_RAW_DATA, $ivRaw);
  if (!is_string($pass)) {
    throw new RuntimeException('Decrypt failed');
  }
  return $pass;
}

function wp_credentials(array $cfg, string $ref): array {
  $paths = tenant_paths($cfg, $ref);
  if (!is_file($paths['wpJson'])) {
    throw new RuntimeException('Tenant not connected to WordPress');
  }
  $data = content_read_json($paths['wpJson']);
  if (!is_array($data)) {
    throw new RuntimeException('Invalid wp.json data');
  }
  $wpUrl = trim((string)($data['wp_url'] ?? ''));
  $wpUser = trim((string)($data['wp_user'] ?? ''));
  if ($wpUrl === '' || $wpUser === '') {
    throw new RuntimeException('WordPress credentials incomplete');
  }
  $masterKey = wp_master_key(wp_master_key_file($cfg));
  $wpPass = wp_decrypt_password($data, $masterKey);

  return [
    'url' => $wpUrl,
    'user' => $wpUser,
    'pass' => $wpPass,
  ];
}

function wp_publish_post(array $credentials, array $post): array {
  $url = rtrim($credentials['url'], '/');
  if ($url === '') {
    throw new RuntimeException('Invalid WordPress URL');
  }
  $endpoint = $url . '/wp-json/wp/v2/posts';

  $ch = curl_init($endpoint);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => [
      'Content-Type: application/json',
      'Accept: application/json',
      'Authorization: Basic ' . base64_encode($credentials['user'] . ':' . $credentials['pass']),
    ],
    CURLOPT_POSTFIELDS => json_encode($post, JSON_UNESCAPED_SLASHES),
    CURLOPT_TIMEOUT => 30,
  ]);

  $body = curl_exec($ch);
  $error = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);

  if ($body === false) {
    return ['ok' => false, 'http' => $code, 'error' => $error ?: 'curl_failed', 'body' => ''];
  }
  return ['ok' => ($code >= 200 && $code < 300), 'http' => $code, 'error' => '', 'body' => (string)$body];
}
