<?php

declare(strict_types=1);

require_once __DIR__ . '/../lib/Env.php';
require_once __DIR__ . '/../lib/Db.php';
require_once __DIR__ . '/../lib/HtmlSanitizer.php';

Env::load(__DIR__ . '/../.env');

$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($id <= 0) {
    http_response_code(400);
    echo 'Invalid draft ID';
    exit;
}

$pdo = Db::getConnection();
$stmt = $pdo->prepare('SELECT * FROM drafts WHERE id = :id');
$stmt->execute([':id' => $id]);
$draft = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$draft) {
    http_response_code(404);
    echo 'Draft not found';
    exit;
}

$imageSrc = '';
if (!empty($draft['image_path']) && file_exists($draft['image_path'])) {
    $imageData = file_get_contents($draft['image_path']);
    if ($imageData !== false) {
        $imageSrc = 'data:image/png;base64,' . base64_encode($imageData);
    }
}

$title = $draft['title'] ?: 'Draft Preview';
$metaDescription = $draft['meta_description'] ?? '';
$contentHtml = sanitizeHtml($draft['content_html'] ?? '');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($title, ENT_QUOTES, 'UTF-8') ?></title>
    <meta name="description" content="<?= htmlspecialchars($metaDescription, ENT_QUOTES, 'UTF-8') ?>">
    <style>
        body {
            font-family: system-ui, sans-serif;
            max-width: 960px;
            margin: 2rem auto;
            padding: 0 1rem;
            line-height: 1.6;
        }

        header img {
            max-width: 100%;
            height: auto;
            display: block;
            margin-bottom: 1rem;
        }

        article h2 {
            margin-top: 2rem;
        }
    </style>
</head>
<body>
    <header>
        <p>Status: <?= htmlspecialchars($draft['status'] ?? 'unknown', ENT_QUOTES, 'UTF-8') ?></p>
        <h1><?= htmlspecialchars($title, ENT_QUOTES, 'UTF-8') ?></h1>
        <p><?= htmlspecialchars($metaDescription, ENT_QUOTES, 'UTF-8') ?></p>
        <?php if ($imageSrc !== ''): ?>
            <img src="<?= $imageSrc ?>" alt="<?= htmlspecialchars($title, ENT_QUOTES, 'UTF-8') ?>">
        <?php endif; ?>
    </header>
    <article>
        <?= $contentHtml ?>
    </article>
</body>
</html>
